# Web Page Agent Interaction Manual

## Overview

This manual provides a comprehensive set of rules for a web page agent to interact with various elements and complete tasks within a web environment. The rules cover navigation, game strategies, email and terminal interactions, form handling, and step optimization.

## Navigation and Search

### Introduction

This category includes rules for navigating search results and handling pagination.

#### Included Rules

- **rule_0 (type="Useful Helper Method")**: If the idx of the target search result exceeds 3 (the number of results per page), use 'turn_to_next_page' method as shown in the example. For example, 
  
  ```python
  # Turn to the next page.
  def turn_to_next_page(agent):
    next_page_xpath = f"//*[@id='pagination']/li[@class='page-item next']"
    html_string = agent.click_xpath(next_page_xpath)
	  return html_string
  ```
  
- **rule_1 (type="Success Process")**: If the task requires using a search engine, first click `<input>` textbox, type the target string, and then click "Search" button. If the target index exceeds 3, use 'turn_to_next_page' in rule_0 to turn the page. Otherwise, click the target result on the current page. For example, 
  
  ```python
  # For example, to search "Tora" and click the 2nd result:
  # [Step 1] Find <input> textbox and type "Tora".
  # [Step 2] Click the search button.
  # [Step 3] Click the target result on the current page if it's within the first three results.
  # [Optional Step] Use 'turn_to_next_page' method to turn the page if the target index exceeds 3 and click the target result.
  ```


- **rule_2 (type="Success Process")**: To click on a search result that is not on the first page, navigate to the correct page using the 'turn_to_next_page' method in rule_0 and then verify the actual `data-result` values before using the correct XPath that reflects the indexing of search results. Ensure the XPath reflects the actual `data-result` values observed on the page, rather than assuming continuous indexing. For example, 
  
  ```python
  # [Step 1] Click the input textbox and type the search term.
  # [Step 2] Click the search button.
  # [Step 3] Use 'turn_to_next_page' method to navigate to the correct page.
  # [Step 4] Verify the actual `data-result` values before clicking on the desired search result.
  # Incorrect XPath assumption: result_xpath = f"//*[@id='page-content']//a[@data-result='0']"
  # Corrected XPath after verification: result_xpath = f"//*[@id='page-content']//a[@data-result='4']"
  ```


## Game Strategy and Interaction

### Introduction

Rules in this category provide strategies for playing games like tic-tac-toe and interacting with game elements.

#### Included Rules

- **rule_3 (type="Useful Helper Method")**: When playing tic-tac-toe, use the 'click_cell' helper method to click on a cell by its ID. For example, 
	
	```python
  # Helper method to click on a cell in the tic-tac-toe board
  def click_cell(agent, cell_id):
    cell_xpath = f"//*[@id='{cell_id}']"
	  return agent.click_xpath(cell_xpath)
	```
	
- **rule_4 (type="Success Process")**: To win a game of tic-tac-toe as 'X', follow a strategic sequence of moves, starting with the center, then corners, and finally completing a line based on the opponent's moves. Ensure to verify the winning condition before making the final move. Additionally, always be prepared to make defensive moves to block the opponent's potential winning lines. **Prioritize blocking the opponent's immediate winning opportunities if they arise.** Use helper methods 'parse_board_state' and 'find_move' to assess the board and determine the best move. For example, 
	
	```python
  # Helper function to parse the board and find the state of each cell
  def parse_board_state(html_string):
	  # ... (existing code)
  
  # Helper function to find a winning or blocking move
  def find_move(board_state, mark):
	    # ... (existing code)
	
	# ... (rest of the success process as in the successful code)
	```

## Email Interaction

### Introduction

This category contains rules for performing email-related tasks such as forwarding.

#### Included Rules

- **rule_5 (type="Success Process")**: When forwarding an email, use the 'forward_email' helper method to encapsulate the process, which involves opening the email, clicking the forward option, entering the recipient's name, and sending the email. Be aware of dynamic content changes such as new HTML elements for forwarding options and recipient fields. For example, 
	
  ```python
  # Helper method to forward an email to a specified recipient
  def forward_email(agent, sender_name, recipient_name):
    # [Step 1] Locate the email thread with the sender's name.
    sender_email_xpath = f"//*[@class='email-sender' and text()='{sender_name}']"
    html_string = agent.click_xpath(sender_email_xpath)
	  # Pause here to ensure that the email interface has opened and the forward option is available.
  
    # [Step 2] Click the "Forward" option within the opened email.
    forward_xpath = "//*[@class='email-forward']"
    html_string = agent.click_xpath(forward_xpath)
	  # Pause here to ensure that the forwarding interface is visible and the recipient field is available.
  
    # [Step 3] Type the recipient's name into the recipient field.
    recipient_field_xpath = "//*[@class='forward-sender']"
    html_string = agent.click_xpath(recipient_field_xpath)
	  html_string = agent.type(recipient_name)
  
    # [Step 4] Click the send button to forward the email.
    send_button_xpath = "//*[@id='send-forward']"
    html_string = agent.click_xpath(send_button_xpath)
	  # No need to pause here as this is the final action to complete the task.
	```
- **rule_6 (type="Success Process")**: When interacting with an email, locate the email by the sender's name and click the appropriate icon (star to mark as important, trash to delete) within the email thread. For example, 
  
  ```python
  # [Step 1] Locate the email thread with the sender 'Stepha'.
  stepha_email_xpath = "//*[@class='email-sender' and text()='Stepha']"
  
  # [Step 2] Find the trash icon within the located email thread.
  trash_icon_xpath = f"{stepha_email_xpath}/ancestor::div[contains(@class, 'email-thread')]//*[@class='trash']"
  
  # [Step 3] Click the trash icon to delete the email.
  html_string = agent.click_xpath(trash_icon_xpath)
  ```

## Terminal Interaction

### Introduction

Rules for interacting with a simulated terminal on a web page.

#### Included Rules

- **rule_7 (type="Success Process")**: When interacting with a simulated terminal on a web page, type commands directly without attempting to click on any elements to focus. Before using 'list_files_with_extension' or 'delete_file_with_extension', verify that these helper methods are defined in the current environment. If not, proceed with manual command input. For example,
	
  ```python 
  # [Step 1] Type the command to list all files with the specified extension.
  list_files_command = "ls *.extension"
  agent.type(list_files_command)
	agent.press_key("enter")
  
  # [Step 2] Observe the output to identify the file name.
  # [Step 3] Type the command to delete the identified file.
  delete_file_command = "rm filename.extension"
  agent.type(delete_file_command)
	agent.press_key("enter")
  
  # Helper method to list files with a specified extension in a simulated terminal environment.
  def list_files_with_extension(agent, extension):
    list_files_command = f"ls *{extension}"
    agent.type(list_files_command)
	  agent.press_key("enter")
  
  # Helper method to delete a file with a specified extension in a simulated terminal environment.
  def delete_file_with_extension(agent, filename_with_extension):
    delete_file_command = f"rm {filename_with_extension}"
    agent.type(delete_file_command)
    agent.press_key("enter")
  # Incorrect approach: Attempting to use an undefined helper method
	list_files_with_extension(agent, ".png")  # This will fail if the method is not defined
  
  # Correct approach: Directly type the command into the terminal
  list_files_command = "ls *.png"
  agent.type(list_files_command)
	agent.press_key("enter")
	```

## Form Interaction

### Introduction

Rules for interacting with web forms, including login processes and element state verification.

#### Included Rules

- **rule_8 (type="Corrected Error")**: After any action that could potentially change the state of form elements, such as closing a popup or alert, verify that the input fields and buttons are enabled and interactable before proceeding. This ensures that the elements are ready for interaction and prevents errors related to disabled elements. Always re-verify the interactability of form elements after closing popups or other UI elements that may affect their state. For example, 
	
  ```python 
  # Incorrect approach: Not verifying the state of form elements after closing a popup.
  # Correct approach: After closing a popup or alert, re-verify that the input fields and buttons are enabled and interactable before proceeding.
  # [Step 1] Close the popup by clicking the "Cancel" button.
  # [Step 2] Re-verify that the input fields and login button are enabled and interactable.
	# [Step 3] Proceed with typing the password and clicking the login button.
	```
	
- **rule_9 (type="Success Process")**: When logging into a web form, click on the username input field and type the username, then click on the password input field and type the password. Before clicking on the login button to submit the form, verify that the input fields and button are enabled and interactable. Additionally, after closing a popup or other UI elements that may disable form elements, re-verify that the input fields and button are enabled and interactable before proceeding. For example, 
	
  ```python 
  # [Step 1] Click on the username input field and type the username.
  # [Step 2] Click on the password input field and type the password.
  # [Verification Step] Verify that the input fields and login button are enabled and interactable.
  # [Step 3] Click on the login button to submit the form.
	# [Additional Verification Step] After closing a popup or other UI elements that may disable form elements, re-verify that the input fields and button are enabled and interactable before proceeding.
	```


## Guessing Game Strategy

### Introduction

Strategies for engaging in and succeeding at guessing games.

#### Included Rules

- **rule_10 (type="Success Process")**: When engaging in a guessing game, initialize a guess to the middle of the range, submit it, and use feedback to iteratively adjust the guess. Ensure to clear the input field before each new guess. For example, 
	
	```python 
	# [Step 1] Initialize the guess to the middle of the range (0-9), which is 4.
	# [Step 2] Click on the input field and type the initial guess.
	# [Step 3] Click the "Submit" button to submit the guess.
	# Pause here to observe the feedback and plan the next steps based on it.
	# Based on the feedback, if the number is lower than the guess, adjust the upper bound and guess again.
	# If the number is higher than the guess, adjust the lower bound and guess again.
	# Repeat steps 4-7 as necessary, adjusting the guess based on the feedback.
	# [Step 4] Clear the input field before typing the new guess.
	# [Step 5] Adjust the guess to the middle of the new range, which is 7.
	# [Step 6] Click on the input field and type the new guess.
	# [Step 7] Click the "Submit" button to submit the new guess.
	# Pause here to observe the feedback and plan the next steps based on it.
	# If the feedback indicates the guess is correct, the task is completed.
	# If not, continue adjusting the guess based on the feedback and repeat steps 8-11.
	# Since the feedback indicates the number is higher than 8, the next guess must be 9.
	# [Step 8] Clear the input field before typing the new guess.
	# [Step 9] Since the number is higher than 8, the next guess must be 9.
	# [Step 10] Click on the input field and type the final guess.
	# [Step 11] Click the "Submit" button to submit the final guess.
	# This should complete the task as 9 is the only number left in the range.
	```
	
- **rule_11 (type="Useful Helper Method")**: Use the 'clear_input_field' method to clear the input field before typing a new guess in a guessing game, tailoring the number of backspace presses to the actual length of the input to conserve steps. For example, 
	
	```python 
  # Helper method to clear the input field
  def clear_input_field(agent, input_xpath, current_value):
      agent.click_xpath(input_xpath)
      for _ in range(len(str(current_value))):  # Clear the current value
	        agent.press_key("backspace")
	# Incorrect approach: Not clearing the input field before typing the new guess
	# Corrected approach: Clear the input field before typing the new guess
	# Inefficient clearing of the input field:
  for _ in range(10):  # Clear the current value (0-9)
	    agent.press_key("backspace")
	# This approach may lead to exceeding the step limit.
	```

## Step Optimization

### Introduction

Rules that focus on minimizing the number of steps taken to complete tasks, especially important in environments with step limits.

#### Included Rules

- **rule_12 (type="Special Phenomena/Mechanism")**: In environments with a step limit, the agent must optimize each action to minimize the number of steps used to avoid exceeding the limit.
